<?php
class smsFlySDK
{
    // Конфигурация API
    private $baseurl;
    private $currency;
    private $apikey;
    private $balance;
    private $lastactionstatus = true;

    // Источники сообщений
    private $sourceList = [];
    private $sourceSMS;
    private $sourceViber;

    // Версия приложения
    private $appversion = 'smsflySDK 1.000';

    /**
     * Конструктор класса
     * @param string $apikey
     */
    public function __construct($apikey)
    {
        $this->apikey = $apikey;
        $this->detectBaseUrl();
        $this->setSources();
    }

    /**
     * Автоматическое определение базового URL и валюты
     */
    private function detectBaseUrl()
    {
        $urls = [
            'https://sms-fly.pl/api/v2/api.php' => 'PLN',
            'https://sms-fly.ua/api/v2/api.php' => 'UAH',
        ];

        foreach ($urls as $url => $currency) {
            $this->baseurl = $url;
            $this->lastactionstatus = true;
            if ($this->checkApiAvailability()) {
                $this->currency = $currency;
                break;
            }
        }
    }

    /**
     * Проверка доступности API
     * @return bool
     */
    private function checkApiAvailability()
    {
        $params = ["action" => "GETBALANCE"];
        $resault = $this->apiquery($params);
        $response =  $resault['response'];
        return !empty($response) && isset($response['data']['balance']);
    }

    /**
     * Установка имен отправителей для SMS и Viber
     * @param string|null $sourceSMS
     * @param string|null $sourceViber
     */
    public function setSources($sourceSMS = null, $sourceViber = null)
    {
        $this->sourceSMS = $sourceSMS ?: '';
        $this->sourceViber = $sourceViber ?: '';
    }
    public function setSourcesSms($sourceSMS = null)
    {
        $this->sourceSMS = $sourceSMS ?: '';
    }
    public function setSourcesViber($sourceViber = null)
    {
        $this->sourceViber = $sourceViber ?: '';
    }

    /**
     * Геттеры для динамических параметров
     */
    public function __get($name)
    {
        switch ($name) {
            case 'auth':
                return $this->lastactionstatus;

            case 'names':
                return $this->getSourceNames();

            case 'balance':
                return $this->getBalance();

            default:
                return null;
        }
    }




    public function tryGetBalance()
    {
        $params = ["action" => "GETBALANCE"];
        $resault = $this->apiquery($params); // Запрос к API
        return $resault;
    }



    /**
     * Получение текущего баланса
     * @return int
     */
    public function getBalance()
    {
        $resault = $this->tryGetBalance();
        $response = $resault['response'];
        if (isset($response['data']['balance'])) {
            return $response['data']['balance']; // Возвращаем баланс
        }
        return 'Error: Unable to fetch balance';
    }

    /**
     * Отправка SMS-сообщения
     * @param string $phone
     * @param string $text
     * @return array
     */
    public function sendSMS($phone, $text)
    {
        if (empty($this->sourceSMS)) {
            return self::craftError('INVSOURCE', 'empty SMS source');
        }

        $recipient = self::checkPhone($phone);
        if (!$recipient) {
            return self::craftError('INVROUTE', $phone);
        }

        $params = [
            "action" => "SENDMESSAGE",
            "data" => [
                "recipient" => $recipient,
                "channels" => ["sms"],
                "sms" => [
                    "source" => $this->sourceSMS,
                    "text" => htmlspecialchars($text),
                ],
            ],
        ];

        error_log(json_encode($params));


        return $this->apiquery($params);
    }

    /**
     * Отправка Viber-сообщения
     * @param array $params
     * @return array
     */
    public function sendViber($params)
    {
        if (empty($this->sourceViber)) {
            return self::craftError('INVSOURCE', 'empty Viber source');
        }

        $recipient = self::checkPhone($params['phone'] ?? '');
        if (!$recipient) {
            return self::craftError('INVROUTE', $params['phone']);
        }

        $data = [
            "recipient" => $recipient,
            "channels" => ['viber'],
            "viber" => [
                "source" => $this->sourceViber,
                "text" => htmlspecialchars($params['text']),
                /*"button" => [
                    "caption" => htmlspecialchars($params['btext']),
                    "url" => $params['blink'],
                ],
                "image" => $params['image'],*/
            ],
        ];

        if (!empty($params['btext']) && !empty($params['blink'])) {
            $data['viber']['button'] = ["caption" => htmlspecialchars($params['btext']), "url" => $params['blink']];
        }
        if (!empty($params['image'])) {
            $data['viber']['image'] = $params['image'];
        }

        if (!empty($params['alttext']) && !empty($this->sourceSMS)) {
            $data['channels'][] = 'sms';
            $data['sms'] = [
                "source" => $this->sourceSMS,
                "text" => htmlspecialchars($params['alttext']),
            ];
        }

        return $this->apiquery(["action" => "SENDMESSAGE", "data" => $data]);
    }

    /**
     * Выполнение запроса к API
     * @param array $params
     * @return mixed
     */
   /* private function apiquery(array $params)
    {
        $params['auth'] = ['key' => $this->apikey, 'appversion' => $this->appversion];

        error_log(json_encode($params));

        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->baseurl,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => ["Content-Type: application/json", "Accept: application/json"],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_POSTFIELDS => json_encode($params, JSON_UNESCAPED_UNICODE),
        ]);
        $result = curl_exec($ch);
        curl_close($ch);
        
        error_log(json_encode($result));

        return $result ? json_decode($result, true) : false;
    }*/


    /**
     * Выполнение запроса к API
     * @param array $params
     * @return array
     */
    private function apiquery(array $params)
    {
        // Добавляем данные авторизации
        $params['auth'] = [
            'key' => $this->apikey,
            'appversion' => $this->appversion,
        ];

        // Преобразуем параметры в JSON
        $jsonRequest = json_encode($params, JSON_UNESCAPED_UNICODE);

        // Логирование запроса
        //error_log('Request: ' . $jsonRequest);

        // Инициализируем cURL
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $this->baseurl,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => ["Content-Type: application/json", "Accept: application/json"],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_POSTFIELDS => $jsonRequest,
        ]);

        // Выполняем запрос
        $rawResponse = curl_exec($ch);

        // Обрабатываем ошибки cURL, если они возникли
        $curlError = curl_error($ch);
        curl_close($ch);

        // Логирование ответа
        error_log('Response: ' . $rawResponse);

        // Если была ошибка cURL
        if ($curlError) {
            return [
                'request' => $params,
                'response' => [
                    'error' => true,
                    'message' => 'cURL Error: ' . $curlError,
                ],
            ];
        }

        // Декодируем ответ
        $response = $rawResponse ? json_decode($rawResponse, true) : null;

        return [
            'request' => $params, // Отправленные данные
            'response' => $response ?? ['error' => true, 'message' => 'Invalid JSON response'], // Ответ от API
        ];
    }

    /**
     * Проверка корректности телефона
     * @param string $number
     * @return string|bool
     */
    static function checkPhone($number)
    {
        $number = preg_replace('/\D+/', '', $number);
        return strlen($number) > 8 ? $number : false;
    }

    /**
     * Создание ошибки
     * @param string $code
     * @param string $description
     * @return array
     */
    static function craftError($code, $description)
    {
        return ['success' => 0, 'error' => ['code' => $code, 'description' => $description]];
    }

    /**
     * Получает текущую валюту системы.
     *
     * Метод определяет текущую валюту, используя настройки или переданные параметры.
     * 
     * @param string|null
     * 
     */
    public function getCurrency()
    {
        return $this->currency;
    }





     /**
     * Получение доступных имен отправителей
     * @return array
     */
    private function getSourceNames()
    {
        if (empty($this->sourceList)) {
            $params = ["action" => "GETSOURCES", "data" => ["channels" => ["sms", "viber"]]];
            $resault = $this->apiquery($params);
            $response = $resault['resault'];

            $this->sourceList['sms'] = $response['data']['sms'] ?? [];
            $this->sourceList['viber'] = $response['data']['viber'] ?? [];
        }
        return $this->sourceList;
    }


    /**
     * Получение доступных ресурсов (имен источников) для SMS и Viber
     *
     * @return array
     */
    public function getSources()
    {
        $params = [
            "action" => "GETSOURCES",
            "data" => [
                "channels" => ["sms", "viber"]
            ]
        ];

        $resault = $this->apiquery($params); // Выполнение запроса через метод apiquery
        $response =  $resault['response'];
        // Проверяем, успешен ли запрос
        error_log(json_encode($response));

        if (isset($response['success']) && $response['success'] == 1 && isset($response['data'])) {
            $this->sourceList['sms'] = $response['data']['sms'] ?? [];
            $this->sourceList['viber'] = $response['data']['viber'] ?? [];
            return $response['data']; // Возвращаем данные о доступных источниках
        }

        // Если запрос завершился неудачей, логируем и возвращаем ошибку
        error_log("Error fetching sources: " . json_encode($response));
        return [
            'error' => 'Unable to fetch sources',
            'details' => $response ?? 'No response from API'
        ];
    }

}
