<?php
require_once 'smsFlySDK.php'; // Подключаем SDK для работы с API SMS-fly

/**
 * Класс SmsFlyOrder для отправки SMS при создании нового заказа.
 * Использует SDK smsFlySDK для интеграции с сервисом SMS-fly.
 */
class SmsFlyOrder extends Module
{
    /**
     * @var smsFlySDK $smsClient Экземпляр клиента SDK smsFlySDK
     */
    private $smsClient;

    /**
     * @var string $logDir Директория для хранения логов
     */
    private $logDir = '../modules/smsflyorder/logs/';

    /**
     * @var string $logFile Название файла для логов
     */
    private $logFile = 'smsfly.log';

    /**
     * Конструктор модуля.
     * Инициализация параметров и метаданных модуля.
     */
    public function __construct()
    {
        $this->name = 'smsflyorder';
        $this->tab = 'emailing';
        $this->version = '1.3.0';
        $this->author = 'SMS-fly Dev';
        $this->displayName = $this->l('SMS-fly Order Module');
        $this->description = $this->l('Sending SMS using the SMS-fly service');
        $this->bootstrap = true;
        $this->need_instance = 0;
        $this->ps_versions_compliancy = ['min' => '1.6', 'max' => _PS_VERSION_];
        $this->module_key = '331499a19c06d6d7b094b87401d80de3';
        parent::__construct();
    }

    /**
     * Установка модуля: регистрация хуков и сохранение настроек.
     *
     * @return bool
     */
    /*public function install()
    {
        // Регистрация хука и сохранение конфигурации
        $installConfig = Configuration::updateValue('SF_APIKEY', '') &&
            Configuration::updateValue('SF_CHANNEL', '') &&
            Configuration::updateValue('SF_SOURCE', 'InfoCentr') &&
            Configuration::updateValue('SF_USER_ON', 0) &&
            Configuration::updateValue('SF_ADMIN_ON', 0) &&
            Configuration::updateValue('SF_ADMIN_NUM', '') &&
            Configuration::updateValue('SF_ATPL', 'New order №{NUM}. Sum - {SUM}. City - {CITY}') &&
            Configuration::updateValue('SF_UTPL', 'Dear {FIRSTNAME}, your order №{NUM} is saved. Total - {SUM}');

        // Создание таблицы в базе данных
        $sql = "
        CREATE TABLE IF NOT EXISTS `" . _DB_PREFIX_ . "smsfly_logs` (
               `id` INT AUTO_INCREMENT PRIMARY KEY,
                `type` VARCHAR(10) NOT NULL,              -- Тип сообщения (SMS или Viber)
                `sender_name` VARCHAR(255) NOT NULL,      -- Имя отправителя
                `status` VARCHAR(50) NOT NULL,            -- Статус отправки (отправлено или ошибка)
                `request_details` TEXT,                   -- Детали запроса (в JSON-формате)
                `response_details` TEXT,                  -- Детали ответа (в JSON-формате)
                `created_at` DATETIME DEFAULT CURRENT_TIMESTAMP -- Дата и время создания записи
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ";

        $createTable = Db::getInstance()->execute($sql);

        // Регистрация хука
        $registerHook = $this->registerHook('newOrder');

        return $installConfig && $createTable && $registerHook;
    }*/

    public function install()
    {
        return parent::install() &&
            $this->registerHook('newOrder') &&
            $this->registerHook('actionOrderStatusPostUpdate') &&
            Configuration::updateValue('SF_APIKEY', '') &&
            Configuration::updateValue('SF_CHANNEL', '') &&
            Configuration::updateValue('SF_SOURCE', 'InfoCentr') &&
            Configuration::updateValue('SF_SOURCE_VIBER', 'InfoCentr') &&
            Configuration::updateValue('SF_USER_ON', 0) &&
            Configuration::updateValue('SF_ADMIN_ON', 0) &&
            Configuration::updateValue('SF_ADMIN_NUM', '') &&
            Configuration::updateValue('SF_ATPL', 'New order №{NUM}. Sum - {SUM}. City - {CITY}') &&
            Configuration::updateValue('SF_UTPL', 'Dear {FIRSTNAME}, your order №{NUM} is saved. Total - {SUM}') &&
            Configuration::updateValue('SF_STATUS_TEMPLATES', '{}') &&
            $this->createLogsTable();
    }

    /**
     * Удаление модуля: удаление настроек из базы данных.
     *
     * @return bool
     */
    public function uninstall()
    {
        return parent::uninstall() &&
            Configuration::deleteByName('SF_APIKEY') &&
            Configuration::deleteByName('SF_CHANNEL') &&
            Configuration::deleteByName('SF_SOURCE') &&
            Configuration::deleteByName('SF_SOURCE_VIBER') &&
            Configuration::deleteByName('SF_USER_ON') &&
            Configuration::deleteByName('SF_ADMIN_ON') &&
            Configuration::deleteByName('SF_ADMIN_NUM') &&
            Configuration::deleteByName('SF_ATPL') &&
            Configuration::deleteByName('SF_UTPL') &&
            Configuration::deleteByName('SF_STATUS_TEMPLATES') &&
            $this->dropLogsTable();
    }

    /**
     * Хук на создание нового заказа.
     * Отправляет SMS пользователю и администратору при создании нового заказа.
     *
     * @param array $params Параметры заказа
     */
    public function hooknewOrder($params)
    {
        // Логирование начала работы хука
        error_log('hooknewOrder triggered');
        error_log('Parameters: ' . print_r($params, true));

        $this->initializeSmsClient();
        /*$apikey = Configuration::get('SF_APIKEY');
        $source = Configuration::get('SF_SOURCE');
        $this->smsClient = new smsFlySDK($apikey);
        $this->smsClient->setSources($source);*/

        // Получение данных о заказе
        $cart = new Cart($params['cart']->id);
        $address = new Address($params['cart']->id_address_delivery);
        $currency = new Currency($params['cart']->id_currency);
        $orderId = $params['order']->id;
        $total = $cart->getOrderTotal() . ' ' . $currency->sign;

        // Шаблонные замены
        $placeholders = [
            '{FIRSTNAME}' => $address->firstname,
            '{LASTNAME}' => $address->lastname,
            '{CITY}' => $address->city,
            '{NUM}' => $orderId,
            '{SUM}' => $total,
        ];

        // Отправка SMS администратору
        if (Configuration::get('SF_ADMIN_ON')) {
            $adminNum = Configuration::get('SF_ADMIN_NUM');
            //$adminPhone = ConfigurationCore::get('SF_ADMIN_NUM');
            $adminText = strtr(Configuration::get('SF_ATPL'), $placeholders);
            $this->sendSms($adminNum, $adminText);

        }

        // Отправка SMS пользователю
        if (Configuration::get('SF_USER_ON')) {
            $userPhone = $address->phone_mobile ?: $address->phone;
            $userText = strtr(Configuration::get('SF_UTPL'), $placeholders);
            $this->sendSms($userPhone, $userText);
        }
    }

    /**
     * Хук на смену статуса заказа. Отправляет SMS клиенту по шаблону, привязанному к статусу.
     *
     * @param array $params ['id_order' => int, 'newOrderStatus' => OrderState|int]
     */
    public function hookActionOrderStatusPostUpdate($params)
    {
        $idOrder = isset($params['id_order']) ? (int) $params['id_order'] : 0;
        if (!$idOrder && isset($params['order']) && is_object($params['order'])) {
            $idOrder = (int) $params['order']->id;
        }
        if (!$idOrder) {
            return;
        }

        $newStatus = isset($params['newOrderStatus']) ? $params['newOrderStatus'] : (isset($params['new_order_status']) ? $params['new_order_status'] : null);
        $idState = is_object($newStatus) ? (int) $newStatus->id : (int) $newStatus;
        if (!$idState) {
            return;
        }

        $templates = $this->getOrderStatusTemplatesConfig();
        if (empty($templates[$idState]['on']) || empty($templates[$idState]['tpl'])) {
            return;
        }

        $this->initializeSmsClient();
        $order = new Order($idOrder);
        $address = new Address($order->id_address_delivery);
        $phone = $address->phone_mobile ?: $address->phone;
        if (empty($phone)) {
            return;
        }

        $placeholders = $this->getOrderPlaceholders($idOrder);
        $text = strtr($templates[$idState]['tpl'], $placeholders);
        $this->sendSms($phone, $text);
    }

    /**
     * Безпечно отримує список статусів замовлень для налаштувань (PS 1.6 / 9).
     *
     * @return array [ ['id_order_state' => int, 'name' => string], ... ]
     */
    private function getOrderStatesForConfig()
    {
        $idLang = (int) $this->context->language->id;
        try {
            if (class_exists('OrderState') && method_exists('OrderState', 'getOrderStates')) {
                $list = OrderState::getOrderStates($idLang);
                if (is_array($list)) {
                    $out = [];
                    foreach ($list as $row) {
                        $id = isset($row['id_order_state']) ? $row['id_order_state'] : ($row['id'] ?? 0);
                        $name = isset($row['name']) ? $row['name'] : ('Status #' . $id);
                        $out[] = ['id_order_state' => (int) $id, 'name' => $name];
                    }
                    return $out;
                }
            }
        } catch (\Throwable $e) {
            // fallback
        }
        $sql = 'SELECT os.id_order_state, osl.name FROM ' . _DB_PREFIX_ . 'order_state os
                LEFT JOIN ' . _DB_PREFIX_ . 'order_state_lang osl ON os.id_order_state = osl.id_order_state AND osl.id_lang = ' . $idLang . '
                WHERE os.deleted = 0 ORDER BY os.id_order_state';
        $rows = Db::getInstance()->executeS($sql);
        if (!is_array($rows)) {
            return [];
        }
        $out = [];
        foreach ($rows as $row) {
            $out[] = [
                'id_order_state' => (int) $row['id_order_state'],
                'name' => !empty($row['name']) ? $row['name'] : ('Status #' . $row['id_order_state']),
            ];
        }
        return $out;
    }

    /**
     * Возвращает конфиг шаблонов по статусам заказа (id_state => ['on' => 0|1, 'tpl' => string]).
     *
     * @return array
     */
    public function getOrderStatusTemplatesConfig()
    {
        $json = Configuration::get('SF_STATUS_TEMPLATES');
        if ($json === false || $json === '') {
            return [];
        }
        $decoded = json_decode($json, true);
        return is_array($decoded) ? $decoded : [];
    }

    /**
     * Плейсхолдеры для подстановки в шаблон по заказу (в т.ч. трекинг).
     *
     * @param int $idOrder
     * @return array [ '{NUM}' => ..., '{TRACKING_NUMBER}' => ..., '{TRACKING_URL}' => ..., ... ]
     */
    protected function getOrderPlaceholders($idOrder)
    {
        $order = new Order($idOrder);
        $address = new Address($order->id_address_delivery);
        $currency = new Currency($order->id_currency);
        $total = $order->getOrdersTotalPaid() . ' ' . $currency->sign;

        $trackingNumber = '';
        $trackingUrl = '';

        $idOrderCarrier = 0;
        if (method_exists($order, 'getIdOrderCarrier')) {
            $idOrderCarrier = (int) $order->getIdOrderCarrier();
        } else {
            $row = Db::getInstance()->getRow(
                'SELECT id_order_carrier FROM ' . _DB_PREFIX_ . 'order_carrier WHERE id_order = ' . (int) $idOrder . ' ORDER BY id_order_carrier DESC LIMIT 1'
            );
            $idOrderCarrier = $row ? (int) $row['id_order_carrier'] : 0;
        }

        if ($idOrderCarrier) {
            $orderCarrier = new OrderCarrier($idOrderCarrier);
            if (!empty($orderCarrier->tracking_number)) {
                $trackingNumber = $orderCarrier->tracking_number;
                $idCarrier = (int) $order->id_carrier;
                if ($idCarrier) {
                    $carrier = new Carrier($idCarrier);
                    if (!empty($carrier->url) && strpos($carrier->url, '@') !== false) {
                        $trackingUrl = str_replace('@', $trackingNumber, $carrier->url);
                    } else {
                        $trackingUrl = $trackingNumber;
                    }
                } else {
                    $trackingUrl = $trackingNumber;
                }
            }
        }

        $shopName = Configuration::get('PS_SHOP_NAME');
        if (!$shopName) {
            $shopName = 'Shop';
        }

        return [
            '{FIRSTNAME}' => $address->firstname,
            '{LASTNAME}' => $address->lastname,
            '{CITY}' => $address->city,
            '{NUM}' => $idOrder,
            '{SUM}' => $total,
            '{SHOP_NAME}' => $shopName,
            '{TRACKING_NUMBER}' => $trackingNumber,
            '{TRACKING_URL}' => $trackingUrl,
        ];
    }

    /**
     * Обработка настроек модуля и выполнение дополнительных действий.
     * Обрабатывает отправку формы для сохранения конфигурации,
     * тестовой отправки SMS и проверки баланса.
     */
    private function processConfig()
    {

        if (!$this->smsClient) {
            $apiKey = Configuration::get('sf_apikey');
            $this->smsClient = new smsFlySDK($apiKey);
        }

        // Проверка и обработка отправки формы настроек модуля
        if (Tools::isSubmit('sf_form_apply')) {
            $this->saveConfig();

            $this->initializeSmsClient();
            $currency = $this->smsClient->getCurrency();

            $this->context->smarty->assign('currency', $currency);
            $this->context->smarty->assign('confirmation', 'ok');
        }

        // Проверка и выполнение тестовой отправки SMS
        if (Tools::isSubmit('sf_form_test')) {
            $this->saveConfig();
            $this->initializeSmsClient();
            $response = json_encode($this->testSms());
            $this->context->smarty->assign('test', 'ok');
        }

        // Проверка и выполнение запроса на получение баланса
        if (Tools::isSubmit('sf_form_balance')) {
            $this->saveConfig();
            $this->initializeSmsClient();
            //$balance = $this->smsClient->getBalance();
            $balance = $this->getBalance();
            $currency = $this->smsClient->getCurrency();
            $this->context->smarty->assign('balance', $balance);
            $this->context->smarty->assign('currency', $currency);
        }
    }

    /**
     * Сохраняет настройки модуля из формы в базу данных.
     */
    private function saveConfig()
    {
        ConfigurationCore::updateValue('SF_APIKEY', Tools::getValue('sf_apikey'));
        ConfigurationCore::updateValue('SF_CHANNEL', Tools::getValue('sf_channel'));
        ConfigurationCore::updateValue('SF_SOURCE', Tools::getValue('sf_source'));
        ConfigurationCore::updateValue('SF_SOURCE_VIBER', Tools::getValue('sf_source_viber'));
        ConfigurationCore::updateValue('SF_USER_ON', Tools::getValue('sf_user_on'));
        ConfigurationCore::updateValue('SF_ADMIN_ON', Tools::getValue('sf_admin_on'));
        ConfigurationCore::updateValue('SF_ADMIN_NUM', Tools::getValue('sf_admin_num'));
        ConfigurationCore::updateValue('SF_ATPL', Tools::getValue('sf_atpl'));
        ConfigurationCore::updateValue('SF_UTPL', Tools::getValue('sf_utpl'));

        $statusOn = Tools::getValue('sf_status_on');
        $statusTpl = Tools::getValue('sf_status_tpl');
        if (!is_array($statusTpl)) {
            $statusTpl = [];
        }
        if (!is_array($statusOn)) {
            $statusOn = [];
        }
        $orderStates = $this->getOrderStatesForConfig();
        $templates = [];
        foreach ($orderStates as $state) {
            $idState = (int) (isset($state['id_order_state']) ? $state['id_order_state'] : $state['id'] ?? 0);
            $tpl = isset($statusTpl[$idState]) ? $statusTpl[$idState] : '';
            $templates[$idState] = [
                'on' => !empty($statusOn[$idState]) ? 1 : 0,
                'tpl' => is_string($tpl) ? trim($tpl) : '',
            ];
        }
        Configuration::updateValue('SF_STATUS_TEMPLATES', json_encode($templates));
    }

    /**
     * Присваивает значения конфигурационных параметров для шаблона Smarty.
     *
     * Этот метод загружает конфигурационные параметры из базы данных и
     * передаёт их в шаблон для отображения в интерфейсе администратора.
     */
    private function assignConfig()
    {

        $this->initializeSmsClient();
        $currency = $this->smsClient->getCurrency();

        $orderStates = $this->getOrderStatesForConfig();
        $statusTemplates = $this->getOrderStatusTemplatesConfig();
        $config = [
            'sf_apikey'       => ConfigurationCore::get('SF_APIKEY'),
            'sf_channel'      => ConfigurationCore::get('SF_CHANNEL'),
            'sf_source'       => ConfigurationCore::get('SF_SOURCE'),
            'sf_source_viber' => ConfigurationCore::get('SF_SOURCE_VIBER'),
            'sf_user_on'      => ConfigurationCore::get('SF_USER_ON'),
            'sf_admin_on'     => ConfigurationCore::get('SF_ADMIN_ON'),
            'sf_admin_num'    => ConfigurationCore::get('SF_ADMIN_NUM'),
            'sf_atpl'         => ConfigurationCore::get('SF_ATPL'),
            'sf_utpl'         => ConfigurationCore::get('SF_UTPL'),
            'tpl_desc'        => $this->l('{NUM} - number order; {SUM} - sum order; {FIRSTNAME} - first name; {LASTNAME} - last name; {CITY} - city customer; {SHOP_NAME} - shop name'),
            'tpl_desc_status' => $this->l('For status templates also: {TRACKING_NUMBER} - tracking number; {TRACKING_URL} - link to track shipment'),
            'hook_desc_new_order'      => $this->l('SMS sent when a new order is created. Set the text for the customer and for the administrator, and enable/disable sending below.'),
            'hook_desc_status_update'  => $this->l('When the order receives one of the statuses below, an SMS with the given text will be sent to the customer. Leave template empty to disable.'),
            'order_states'    => $orderStates,
            'status_templates'=> $statusTemplates,
            'SMS_alfanames'   => $this->getAlfanames("sms"),
            'VIBER_alfanames' => $this->getAlfanames("viber"),
            'currency'        => $currency,
        ];

        // Передаём параметры в шаблон Smarty
        foreach ($config as $key => $value) {
            $this->context->smarty->assign($key, $value);
        }
    }

    /**
     * Отображает страницу настроек модуля в панели администратора PrestaShop.
     *
     * Этот метод обрабатывает сохранение конфигурации, передаёт параметры в шаблон
     * и возвращает HTML-контент для отображения.
     *
     * @return string HTML-контент страницы настроек
     */
    public function getContent()
    {

        $this->context->smarty->assign([
            'module_link' => $this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name,
            'logs_link' => $this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&viewLogs=1',
        ]);

        if (Tools::getValue('viewLogs')) {
            if (Tools::isSubmit('sf_clear_logs')) {
                $this->clearLogs();
                Tools::redirectAdmin($this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&viewLogs=1');
                return '';
            }
            return $this->getLogsPage();
        }

        // Обработка сохранения и других действий
        $this->processConfig();

        // Подготовка данных для отображения в шаблоне
        $this->assignConfig();


        // Возвращаем готовый HTML-контент из файла шаблона (admin, не front)
        return $this->display(__FILE__, 'views/templates/admin/getContent.tpl');
    }

    /**
     * Тестовая отправка SMS для проверки работоспособности модуля.
     *
     * Этот метод отправляет тестовое SMS-сообщение на номер администратора,
     * используя заранее заданный текст.
     */
    private function testSms()
    {
        // Получаем номер телефона администратора из конфигурации
        $adminPhone = ConfigurationCore::get('SF_ADMIN_NUM');

        // Устанавливаем текст для тестового сообщения
        $testMessage = "Prestashop test message.";

        // Вызываем метод отправки SMS
        return $this->sendSms($adminPhone, $testMessage);
    }

    /**
     * Получение текущего баланса
     * @return int
     */
    public function getBalance()
    {
        $this->initializeSmsClient();
        $resault = $this->smsClient->tryGetBalance();
        $response = $resault['response'];
        $request = $resault['request'];
        $request['auth']['key'] = "****";
        $status = 'error';

        if (isset($response['data']['balance'])) {
            $status = 'success';
        }
        $this->addLog('getbalance', 'none', $status, $request, $response);

        if (isset($response['data']['balance'])) {
            return $response['data']['balance']; // Возвращаем баланс
        }
        return 'Error: Unable to fetch balance';
    }

    /**
     * Отправляет SMS, Viber или SMS и Vibe сообщение в зависимости от выбранного канала.
     *
     * Этот метод определяет, какой канал отправки (SMS, Viber или оба) настроен в конфигурации,
     * и вызывает соответствующий метод для отправки сообщения.
     *
     * @param string $phone Номер телефона получателя сообщения.
     * @param string $text Текст сообщения.
     */
    private function SendSms($phone, $text){
        // Получаем какой канал использовать для отправки.
        $chanel = ConfigurationCore::get('SF_CHANNEL');
        // Проверяем выбранный канал отправки
        if ($chanel == 'viber'){
            // Отправляем через Viber
            $this->sendSmsOrderByChanel("viber", $phone, $text);
        }elseif ($chanel == 'sms_and_viber'){
            // Отправляем через оба канала: Viber и SMS
            $this->sendSmsOrderByChanel("viber", $phone, $text);
            $this->sendSmsOrderByChanel("sms", $phone, $text);
        }else{
            // Отправляем только через SMS
            $this->sendSmsOrderByChanel("sms", $phone, $text);
        }
    }

    /**
     * Отправляет сообщение через указанный канал (SMS или Viber) и логирует результат.
     *
     * Этот метод инициализирует клиента SMS-fly, пытается отправить сообщение через
     * выбранный канал, проверяет статус ответа и записывает лог операции.
     *
     * @param string $chanel Канал отправки сообщения ('sms' или 'viber').
     * @param string $phone Номер телефона получателя.
     * @param string $text Текст сообщения.
     */
    private function sendSmsOrderByChanel($chanel, $phone, $text)
    {
        $this->initializeSmsClient();
        $rasault_all = $this->trySendSms($chanel, $phone, $text);
        $request = $rasault_all['request'];
        $request['auth']['key'] = "****";

        $response = $rasault_all['response'];
        $status = "error";

        if (isset($response['success'])){
            if($response['success'] == 1){
                $status = "shipped";
            }
        }

        // Получение имени отправителя из конфигурации
        $sender = Configuration::get($chanel == 'viber' ? 'SF_SOURCE_VIBER' : 'SF_SOURCE');
        $this->addLog($chanel, $sender, $status, $request, $response);
    }

    /**
     * Попытка отправить SMS или Viber сообщение через заданный канал.
     *
     * В зависимости от переданного канала (`viber` или `sms`) метод вызывает соответствующий
     * метод клиента SMS-fly для отправки сообщения. Передача параметров запроса зависит от
     * типа канала.
     *
     * @param string $chanel Канал отправки сообщения. Может быть 'viber' или 'sms'.
     * @param string $phone Номер телефона получателя. Должен быть в международном формате.
     * @param string $text Текст сообщения, которое необходимо отправить.
     *
     * @return array Результат выполнения запроса к API SMS-fly.
     *               Включает данные запроса и ответа.
     *               Формат ответа зависит от реализации клиента SMS-fly.
     *
     * Пример использования:
     * ```
     * $result = $this->trySendSms('viber', '380XXXXXXXXX', 'Ваш заказ отправлен!');
     * if ($result['success'] === 1) {
     *     echo 'Сообщение успешно отправлено';
     * } else {
     *     echo 'Ошибка отправки: ' . $result['error'];
     * }
     * ```
     */

    private function trySendSms($chanel, $phone, $text){
        if ($chanel == 'viber'){
            return $this->smsClient->sendViber( array(
                'phone' => $phone,
                'text' => $text,
            ));
        }else{
            return $this->smsClient->sendSMS($phone, $text);
        }
    }

    /**
     * Логирование сообщений об ошибках в файл.
     *
     * @param string $message Сообщение для записи в лог
     */
    private function logError($message)
    {
        $logPath = $this->logDir . $this->logFile;
        file_put_contents($logPath, date('Y-m-d H:i:s') . ' - ' . $message . "\n", FILE_APPEND);
    }

    /**
     * Инициализация клиента smsFlySDK.
     * 
     * @return smsFlySDK Экземпляр клиента smsFlySDK.
     */
    private function initializeSmsClient()
    {
        $apikey =       Configuration::get('SF_APIKEY'); // Получаем API-ключ из настроек
        $source =       Configuration::get('SF_SOURCE'); // Получаем источник (альфа-имя) из настроек
        $source_viber = Configuration::get('SF_SOURCE_VIBER'); // Получаем источник (альфа-имя) из настроек
        $this->smsClient = new smsFlySDK($apikey); // Инициализируем клиент
        $this->smsClient->setSourcesViber($source_viber); // Устанавливаем источник сообщений viber
        $this->smsClient->setSourcesSms($source);  // Устанавливаем источник сообщений
        return $this->smsClient;
    }

    /**
     * Получение списка доступных имен отправителей (AlfaNames) для SMS.
     *
     * @return array Массив с именами отправителей.
     */
    private function getAlfanames($channel)
    {
        $apiKey =  Configuration::get('SF_APIKEY');
        $this->smsClient = new smsFlySDK($apiKey);

        $sources = $this->smsClient->getSources();
        $alfanames = []; // пустой массив

        if ($channel == "viber"){
            if (isset($sources['viber'])) {
                $alfanames = $sources['viber'];
            }
        }else{
            if (isset($sources['sms'])) {
                $alfanames = $sources['sms'];
            }
        }

        return $alfanames;
    }

    /**
     * Страница логов
     * @return mixed
     */
    private function getLogsPage()
    {
        $this->checkApiKeyAndRemoveLog();

        $logs = $this->getLogData(); // Получаем данные логов из базы

        foreach ($logs as &$log) {
            $token = isset($log['token']) ? trim((string) $log['token']) : '';
            if ($token === '' || $token === 'N/A' || strlen($token) < 7) {
                $log['token'] = 'N/A';
            } else {
                $log['token'] = substr($token, 0, 7) . str_repeat('*', max(0, strlen($token) - 7));
            }
        }

        // Передаем данные в шаблон Smarty
        $this->context->smarty->assign('logs', $logs);
        $this->context->smarty->assign('clear_log_url', $this->context->link->getAdminLink('AdminModules', true) . '&configure=' . $this->name . '&viewLogs=1');
        //$this->context->smarty->assign('logs', json_encode($logs));


        // Подключаем шаблон для отображения логов
        return $this->display(__FILE__, 'views/templates/front/logs.tpl');
    }

    /**
     * Получение данных логов
     * @return array[]
     */
    private function getLogData()
    {
        $sql = 'SELECT * FROM ' . _DB_PREFIX_ . 'smsfly_logs ORDER BY created_at DESC';
        $logs = Db::getInstance()->executeS($sql);

        if (!$logs) {
            return [
                [
                    'type' => 'N/A',
                    'token' => 'N/A',
                    'sender' => 'N/A',
                    'status' => 'error',
                    'request' => 'No log entries found',
                    'response' => 'N/A',
                    'created_at' => date('Y-m-d H:i:s'),
                ],
            ];
        }

        return $logs;
    }

    /**
     * Добавление лога
     * @param $type
     * @param $sender
     * @param $status
     * @param $request
     * @param $response
     *
     * @return void
     */
    public function addLog($type, $sender, $status, $request, $response)
    {
        $apiKey = Configuration::get('SF_APIKEY');
        if ($apiKey === false || $apiKey === null || trim((string) $apiKey) === '') {
            $apiKey = 'N/A';
        } else {
            $apiKey = trim((string) $apiKey);
        }

        $data = [
            'type' => pSQL($type),
            'token' => pSQL($apiKey),
            'sender' => pSQL($sender),
            'status' => pSQL($status),
            'request' => pSQL(json_encode($request, JSON_UNESCAPED_UNICODE)),
            'response' => pSQL(json_encode($response, JSON_UNESCAPED_UNICODE)),
            'created_at' => date('Y-m-d H:i:s'),
        ];


        Db::getInstance()->insert('smsfly_logs', $data);
    }

    /**
     * Создание таблицы логов
     * @return mixed
     */
    private function createLogsTable()
    {
        $sql = '
        CREATE TABLE IF NOT EXISTS `' . _DB_PREFIX_ . 'smsfly_logs` (
            `id` INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,
            `token` VARCHAR(50) NOT NULL,
            `type` VARCHAR(50) NOT NULL,
            `sender` VARCHAR(100) NOT NULL,
            `status` VARCHAR(50) NOT NULL,
            `request` TEXT NOT NULL,
            `response` TEXT NOT NULL,
            `created_at` DATETIME NOT NULL,
            PRIMARY KEY (`id`)
        ) ENGINE=' . _MYSQL_ENGINE_ . ' DEFAULT CHARSET=utf8mb4;
    ';

        return Db::getInstance()->execute($sql);
    }

    /**
     * Очистка таблицы логов
     * @return mixed
     */
    private function dropLogsTable()
    {
        $sql = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'smsfly_logs`;';
        return Db::getInstance()->execute($sql);
    }

    /**
     * Очистка логов: таблица в БД и файл smsfly.log
     */
    public function clearLogs()
    {
        $table = _DB_PREFIX_ . 'smsfly_logs';
        $exists = Db::getInstance()->executeS("SHOW TABLES LIKE '" . $table . "'");
        if (!empty($exists)) {
            Db::getInstance()->execute('TRUNCATE TABLE `' . $table . '`');
        }
        $logFilePath = dirname(__FILE__) . '/logs/' . $this->logFile;
        if (file_exists($logFilePath)) {
            @file_put_contents($logFilePath, '');
        }
    }

    /**
     * Проверяет API-ключ и удаляет таблицу логов, если API-ключ изменен.
     * @return mixed
     */
    public function checkApiKeyAndRemoveLog()
    {
        $apiKey =  Configuration::get('SF_APIKEY');

        // Если текущий API Key пуст, очищаем таблицу
        if (empty($apiKey) || $apiKey == '' || $apiKey == 'N/A') {
            error_log('API Key is not set. Recreating logs table...');
            $this->dropLogsTable();
            $this->createLogsTable();
            return $apiKey;
        }

        // Получаем все токены из таблицы, исключая пустые строки
        $sql = 'SELECT DISTINCT `token` FROM `' . _DB_PREFIX_ . 'smsfly_logs` WHERE `token` != ""';
        $tokens = Db::getInstance()->executeS($sql);

        $tokensDiffer = false;

        if ($tokens) {
            foreach ($tokens as $row) {
                if ($row['token'] !== $apiKey) {
                    $tokensDiffer = true;
                    break;
                }
            }
        } else {
            // Если таблица пуста
            error_log('No tokens found in the logs table. Creating logs table...');
            $this->createLogsTable();
            return $apiKey;
        }

        if ($tokensDiffer) {
            // Если хотя бы один токен отличается, обнуляем таблицу
            error_log('API Key has been changed. Recreating logs table...');
            $this->dropLogsTable();
            $this->createLogsTable();
        } else {
            // Если все токены совпадают
            error_log('All tokens match the current API Key. No changes made.');
        }

        return $apiKey;
    }
}


